/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: SymbolStates.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck;

import scala.tools.nsc._
import scala.collection.mutable.HashMap

/**
 * Representation of the state of a symbol during type inference.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait SymbolStates {
  val global: Global
  import global._
  
  /**
   * Class to store some state information of symbols, used during type inference.
   * @param code Tree containing the AST with which the symbol gets initialized.
   * @param unit Compilation unit this symbol belongs to.
   */
  class SymbolState(val code: Tree, val unit: CompilationUnit) {
    /**
     * Has the symbol already been typed, ie. some type was inferred?
     */
    private var typed = false
    
    /**
     * Set the symbol as being typed.
     */
    def setTyped {
      typed = true
    }
      
    /**
     * Check if the symbol has been typed.
     */
    def isTyped = typed
    
    /**
     * Get the symbol this state belongs to.
     */
    def sym = code.symbol
    
    /**
     * Is this symbol currently locked, ie. inference of its type running?
     */
    private var locked = false
    
    /**
     * Lock the symbol.
     */
    def lock {
      locked = true
    }
    
    /**
     * Unlock the symbol.
     */
    def unlock {
      locked = false
    }
    
    /**
     * Check if the symbol is locked.
     * @return if the symbol is currently locked.
     */
    def isLocked = locked
    
    /**
     * Convert the state information to a string.
     * @return the string representation of this symbol.
     */
    override def toString : String = {
      sym+(if (locked) " [locked]" else "")+(if (typed) " [typed]" else "")+": "+sym.tpe+" => "+code
    }
  }
  
  object SymbolState {
    /**
     * Map to store state information about symbols - used in type inference.
     */
    private var symbolStates = new HashMap[Symbol,SymbolState]
    
    /**
     * Add a new symbol and its state to the state map.
     * @param entry The map entry to add.
     */
    def add(entry: (Symbol,SymbolState)) {
      symbolStates += entry
    }
    
    /**
     * Check if a given symbol already has been typed, ie. a 
     * type has been inferred.
     * @param symbol The symbol which should be checked.
     * @return A pair of a boolean flag if it was typed and the 
     *         state of the symbol if available or null.
     */
    def isTyped(symbol: Symbol) : (Boolean, SymbolState) = {
      if (symbol == null) {
         (false,null) 
      }
      else {
        symbolStates.get(symbol) match {
          case Some(state) => (state.isTyped, state)
          case None        => (false,         null)
        }
      }
    }
    
    /**
     * Update the state of a symbol if it is in the map.
     * @param symbol The symbol whose state should be updated.
     * @param f      The function which does the update of the state.
     */
    private def updateSymbolState(symbol: Symbol, f: SymbolState => Unit) {
      symbolStates.get(symbol) match {
        case Some(state) => f(state)
        case None        => 
      }  
    }
    
    /**
     * Set that a symbol has been typed.
     * @param symbol The symbol whose type got inferred.
     */
    def setTyped(symbol: Symbol) {
      updateSymbolState(symbol, _.setTyped)
    }
    
    /**
     * Lock a given symbol.
     * @param symbol The symbol to lock.
     */
    def lock(symbol: Symbol) {
      updateSymbolState(symbol, _.lock)
    }
    
    /**
     * Unlock a given symbol.
     * @param symbol The symbol to unlock.
     */
    def unlock(symbol: Symbol) {
      updateSymbolState(symbol, _.unlock)
    }
  }
}
