/**
 * UrtHashtable - a Hashtable that uses ints as the keys
 *
 * based on IntHashtable by the ACME Labs:
 * <a href="http://www.acme.com/java/software/Acme.IntHashtable.html">ACME IntHashtable</a>
 *
 * This one is highly optimized to be used for the dynamic checks
 * of the universe type system. (@see UniverseRuntime)
 *
 * scdaniel - Changelog
 * --------------------
 * - removed:
 *	enumeration
 *	all but the default constructor
 *	size()
 *	isEmpty()
 *	contains()
 *	containsKey()
 *	put(Object, Object)
 *	get(Object)
 *	remove(Object)
 *	toString()
 *	clone()
 * - changes to get(), put() and remove()
 *   details are found in the respective javadocs
 */

package org.multijava.universes.rt.impl;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;

import org.multijava.mjc.MjcTokenTypes;
import org.multijava.util.Utils;

public class UrtHashtable {
    // The hash table data.
    private UrtHashtableEntry table[];

    // The total number of entries in the hash table.
    private int count;

    // Rehashes the table when count exceeds this threshold.
    private int threshold;

    // The load factor for the hashtable.
    private float loadFactor;

    // Constructs a new, empty hashtable. A default capacity and load factor
    // is used. Note that the hashtable will automatically grow when it gets
    // full.
    public UrtHashtable () {
	int initialCapacity = 101;
	float loadFactor = 0.75f;

	if ( initialCapacity <= 0 || loadFactor <= 0.0 )
	    throw new IllegalArgumentException();
	this.loadFactor = loadFactor;
	table = new UrtHashtableEntry[initialCapacity];
	threshold = (int) (initialCapacity * loadFactor);
    }

    /**
     *	Gets the record associated with the specified object.
     *	System.identityHashCode() is used as key.
     *
     *	@param	o the object to search for
     *
     *	@return	the element for the object or null if the
     *			object is not defined in the hash table.
     *
     *
     *	scaniel - Changelog
     *	-------------------
     *	- use an Object instead of an int key to get the record 
     *	- return the UrtHashtableEntry instead of a value
     *	- removed tab, which was a reference to table, for simplicity
     *
     *	This is needed, since we need to make sure it is the
     *	right Entry, ie. the one linking to the Object.
     *	And it works, since the int key is System.identitHashCode
     *	of the Object.
     */
    public synchronized UrtHashtableEntry get (Object o) {
	// scdaniel
	int hash = System.identityHashCode(o);
		
	int index = (hash & 0x7FFFFFFF) % table.length;
	for ( UrtHashtableEntry e = table[index]; e != null; e = e.next ) {
	    // scdaniel: last test needed to make sure it's the right one
	    if ( e.hash == hash && e.holds(o) )
		return e;
	}
	return null;
    }

    /**
     *	Gets the record that uses the specified UrtWeakReference
     *	to link to the object it is associated with.
     *
     *	@param	r the UrtWeakReference to search for
     *
     *	@return	the element for the object or null if the
     *			object is not defined in the hash table.
     */
    public synchronized UrtHashtableEntry get (UrtWeakReference r) {
	int hash = r.hash;
		
	int index = (hash & 0x7FFFFFFF) % table.length;
	for ( UrtHashtableEntry e = table[index]; e != null; e = e.next ) {
	    if ( e.hash == hash && e.obj == r )
		return e;
	}
	return null;
    }

    // Rehashes the content of the table into a bigger table.
    // This method is called automatically when the hashtable's
    // size exceeds the threshold.
    protected void rehash () {
	int oldCapacity = table.length;
	UrtHashtableEntry oldTable[] = table;
	int newCapacity = oldCapacity * 2 + 1;
	UrtHashtableEntry newTable[] = new UrtHashtableEntry[newCapacity];
	threshold = (int) (newCapacity * loadFactor);
	table = newTable;
	for ( int i = oldCapacity; i-- > 0; ) {
	    for ( UrtHashtableEntry old = oldTable[i]; old != null; ) {
		UrtHashtableEntry e = old;
		old = old.next;
		int index = (e.hash & 0x7FFFFFFF) % newCapacity;
		e.next = newTable[index];
		newTable[index] = e;
	    }
	}
    }

    /**
     *	Puts the specified element into the hashtable, using
     *	System.identityHashCode() as key.
     *	The element and the entry cannot be null.
     *
     *	@param	o the element to add
     *	@param	e the entry to use for this element
     *
     *	@return	the entry in the hashtable for this object
     *
     *	@exception	NullPointerException If the object or the
     *				entry is equal to null.
     *
     *	scdaniel - Changelog
     *	--------------------
     *	- use the an Object and an UrtHashtableEntry instead
     *	  of a key to allow different entry types
     *	  (see IntHashtableThreadEntry)
     *	- test if we need to rehash earlier, to save some instructions
     *	- also save objects with duplicate hashcodes
     *	- removed tab, which was a reference to table, for simplicity
     */
    public synchronized UrtHashtableEntry put (Object o,
					       UrtHashtableEntry e) {
	// Make sure the entry is not null.
	if ( (o == null) || (e == null) )
	    throw new NullPointerException();

	// scdaniel: do this before the loop, saves some time
	if ( count >= threshold ) {
	    // Rehash the table if the threshold is exceeded.
	    rehash();
	}
		
	// scdaniel
	int hash = System.identityHashCode(o);
		
	int index = (hash & 0x7FFFFFFF) % table.length;
	for ( UrtHashtableEntry i = table[index]; i != null; i = i.next ) {
	    /*
	     * scdaniel: we want to be able to have objects
	     * with duplicate hashCodes, so only return on
	     * really duplicate objects
	     */
	    if ( (i.hash == hash) && i.holds(o) )
		return i;
	}
		
	// Creates the new entry.
	e.hash = hash;
	e.next = table[index];
	table[index] = e;
	++count;
	return e;
    }

    /**
     *	Removes the specified element.
     *	Does nothing if the key is not present.
     *
     *	@param	e the entry that needs to be removed
     *
     *	scdaniel - Changelog
     *	--------------------
     *	- use the entry itself instead of a key
     *	- remove only when the very same (==) element is found
     *	- removed tab, which was a reference to table, for simplicity
     */ 
    public synchronized void remove (UrtHashtableEntry e) {
	int hash = e.hash;
	int index = (hash & 0x7FFFFFFF) % table.length;
	for ( UrtHashtableEntry i = table[index], prev = null;
	      i != null; prev = i, i = i.next ) {
	    if ( i == e) {
		if ( prev != null )
		    prev.next = i.next;
		else
		    table[index] = i.next;
		--count;
	    }
	}
    }
}


/**
 * 	The class for the entries used in UrtHashtable.
 * 
 *	scdaniel - Changelog
 *	--------------------
 *	- removed value, we're using the entry itself to store everything
 *	- removed key, it's equal to hash
 *	- added some attributes used for the ownerHashtable
 * 
 * The fields are accessible directly for speed reasons.
 */
class UrtHashtableEntry {
    int hash;
	UrtHashtableEntry next;
	
	/*
	 *	scdaniel
	 *
	 *	We need the following attributes for our ownerHashtable.
	 */
	UrtWeakReference obj;
	UrtHashtableEntry owner;
	int children = 0;
	
	/**
	 *	Tests if this instance of UrtHashtableEntry holds
	 *	the Object specified. (ie. links to it)
	 * 
	 *	@param	o	the Object to test
	 *	@return	true if this UrtHashtableEntry holds the specified Object
	 */
	public boolean holds (Object o) {
		return ( this.obj.get() == o );
	}
}

/**
 * The class used for entries associated to arrays. Stores the element type
 * in addition to the normal things.
 * 
 * The fields are accessible directly for speed reasons.
 */
class UrtHashtableArrayEntry extends UrtHashtableEntry {
	int elementType;
	
	/**
	 * Constructor that sets the element type of the array this entry is
	 * representing.
	 * 
	 * @param elementType	the element type
	 * 			(one of MjcTokenTypes.LITERAL_peer, MjcTokenTypes.LITERAL_rep)
	 */
	public UrtHashtableArrayEntry (int elementType) {
		this.elementType = elementType;
	}
}

/**
 * This class is used for entries associated to threads.
 * Provides additional information to allow to set the owner in the constructor
 * and therefore associate objects created in the constructor (and fields) with
 * their correct owners.
 * This wouldn't work when setting the owner after the "new" operator.
 * 
 * It also saves the context history needed for static functions.
 * Before a static function is called from a non-static context, the current
 * object is put on top of the stack so inside the static function, the same
 * context can be used. When the static function returns, the top element of
 * the stack is removed again.
 * 
 * The fields are accessible directly for speed reasons.
 */
class UrtHashtableThreadEntry extends UrtHashtableEntry {
    Object currentObject;
    int modifier;
    Object objectClass;

    private UrtStackItem contexts;
	
    /**
     * Constructor that takes the object it will link to as argument to
     * initialize the additional data in case this thread is used to
     * create a universe aware object while it is in a non-universe aware
     * context.
     * 
     * @param currentObject	the object it is to be associated with
     * @param root the root object of the context stack
     */
    public UrtHashtableThreadEntry (Object currentObject, Object root) {
	this.currentObject = currentObject;
	modifier = MjcTokenTypes.LITERAL_peer;
	this.objectClass = null;
	contexts = new UrtStackItem(root, null);
    }
	
    /**
     * Push the current context on the stack.
     * 
     * @param context
     */
    public synchronized void push (Object context) {
	contexts = new UrtStackItem(context, contexts);
    }
	
    /**
     * Remove the topmost stack element.
     */
    public synchronized void pop () {
	Utils.assertTrue(contexts != null);
		
	contexts = contexts.next;
    }
	
    /**
     * Get the current context. Without changing the stack.
     * 
     * @return the topmost stack element
     */
    public Object getCurrentContext () {
	Utils.assertTrue(contexts != null);
	
	return contexts.obj;
    }
}


/**
 * This is a simple stackItem that just stores a reference to an object and
 * a reference to the next item on the stack.
 * 
 * The fields are accessible directly for speed reasons.
 */
class UrtStackItem {
    Object obj;
    UrtStackItem next;
	
    UrtStackItem (Object obj, UrtStackItem next) {
	this.obj = obj;
	this.next = next;
    }
}


/**
 * A class that extends WeakReference and adds the additional attribute
 * hash storing the System.identityHashCode of the object it referes to.
 * This allows to delete the hashtable entry of that object if the object
 * is collected and we are noticed by the reference queue.
 * 
 * The fields are accessible directly for speed reasons.
 */
class UrtWeakReference extends WeakReference {
    // the hashcode of the object we're linking to
    int hash;
	
    // just calls super and sets the hashcode
    UrtWeakReference (Object o) {
	super(o);
	hash = System.identityHashCode(o);
    }

    // just calls super and sets the hashcode
    UrtWeakReference (Object o, ReferenceQueue q) {
	super(o, q);
	hash = System.identityHashCode(o);
    }
}
