/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: CompilationTestFactory.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.test.helpers

import scala.tools.nsc.Main

import scala.testing.SUnit
import SUnit._

/**
 * Produce test cases where a compilation of a given class is attempted and the 
 * result gets compared to the expected number of warnings and errors. 
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait CompilationTestFactory extends TestUtils {
  /**
   * General options to use during compilation.
   */
	val options : List[String]
  
  /**
   * Options which were added when executing the test's main function.
   */
  var dynOptions : List[String] = Nil
  
  /**
   * A list of compiler plugins which should be used during compilation.
   */
  val plugins : List[String]
  
  /**
   * Destination directory where the compiled classes should be stored.
   */
  val destination : String
  
  /**
   * Source directory which is prepended to the given classes.
   */
  val srcdir : String
  
  /**
   * Common options used for all compilations. 
   */
  private val commonOptions : List[String] = List(
      "-deprecation",
      "-unchecked"
  )
  
  /**
   * Create an instance of the testcase using given testdata.
   * @param cls The class which should be compiled.
   * @param expected_errors Number of errors which are actually expected.
   * @param expected_warnings Number of warnings which are actually expected.
   * @param n Some name for the test.
   * @return A testcase.
   */
  def produce(cls: String, expected_errors: Int, expected_warnings: Int, n: String) : CompilationTest = {
    new CompilationTest(List(cls), expected_errors, expected_warnings, n)
  }
  
  /**
   * Create an instance of the testcase using given testdata.
   * @param cls The classes which should be compiled.
   * @param expected_errors Number of errors which are actually expected.
   * @param expected_warnings Number of warnings which are actually expected.
   * @param n Some name for the test.
   * @return A testcase.
   */
  def produce(cls: List[String], expected_errors: Int, expected_warnings: Int, n: String) : CompilationTest = {
    new CompilationTest(cls, expected_errors, expected_warnings, n)
  }
  
  /**
   * Class implementing a <code>TestCase</code> which tries to compile a given class, 
   * and checks the number of errors/warnings generated.
   * @param cls Name of the class including package. Will be prepended by <code>src/examples/scala/</code>.
   * @param expected_errors Number of errors which are actually expected.
   * @param expected_warnings Number of warnings which are actually expected.
   * @param n Name for this test case.
   */
  class CompilationTest(cls: List[String], expected_errors: Int, expected_warnings: Int, n: String) extends TestCase(n) {
    override def runTest() = {
      var sources = cls map {s => srcdir+s.replaceAll("\\.","/")+".scala"}
      val info = "Compiling "+sources+", expecting "+expected_errors+" errors and "+expected_warnings+" warnings."
      val sharps = line('#',info,120)
      println(sharps+"\n"+info+"\n"+sharps)
      var args : List[String] = options ::: dynOptions ::: (plugins map {"-Xplugin:"+_}) ::: commonOptions ::: List("-d",destination) ::: sources 
      scala.tools.nsc.Main.process(args.toArray)
      val errors = scala.tools.nsc.Main.reporter.ERROR.count 
      assertTrue(expected_errors+  " errors expected, "+  errors+  " errors found in "+  n+".scala", errors   == expected_errors)
      val warnings = scala.tools.nsc.Main.reporter.WARNING.count 
      assertTrue(expected_warnings+" warnings expected, "+warnings+" warnings found in "+n+".scala", warnings == expected_warnings)
    }
  }
}
