/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: DefaultUTSMethodSignature.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.rules.default

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.plugin.staticcheck._
import ch.ethz.inf.sct.uts.plugin.staticcheck.common._
import ch.ethz.inf.sct.uts.annotation._

/**
 * Trait implementing type-system dependent behavior of the <code>MethodSignature</code>.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait DefaultUTSMethodSignature[G <: Global] extends UTSMethodSignature[G] {
  self: DefaultTypeAbstraction[G] =>
  import global._
  import extendedType._
  
  /**
   * Factory method to create new <code>MethodSignature</code> instances.
   * @param typeParams Type parameters of this method, if any.
   * @param isPure     Flag if this method is pure.
   * @param returnType Return type of this method.
   * @param paramTypes Method parameter types, if any.
   * @param sym        Symbol of the method.
   */
  def methodSignature(typeParams: List[UType], isPure: Boolean, returnType: UType, paramTypes: List[UType], sym: Symbol) : MethodSignature = 
    new DefaultMethodSignature(typeParams,isPure,returnType,paramTypes,sym)

  
  /**
   * Default implementation of a method signature representation.
   * @param typeParams Type parameters of this method, if any.
   * @param isPure     Flag if this method is pure.
   * @param returnType Return type of this method.
   * @param paramTypes Method parameter types, if any.
   * @param sym        Symbol of the method.
   */
  class DefaultMethodSignature(typeParams: List[UType], isPure: Boolean, returnType: UType, paramTypes: List[UType], sym: Symbol) 
    extends MethodSignature(typeParams,isPure,returnType,paramTypes,sym) {
    
    /**
     * Check if method in <code>this</code> correctly overrides method in <code>that</code>.
     * @param that The method which should be overriden by the method in <code>this</code>.
     * @return if overriding is correct.
     */
    def overrides (that: MethodSignature) : Boolean = {
      // Create mapping for substitution
      val smap = UType(that.sym.owner.tpe) match {
        case nt: NType => nt.typeParams map {
          case tv: TVarID => (tv.tpe,UType(tv.tpe.asSeenFrom(this.sym.owner.tpe,that.sym.owner)))
          case t          => (t.tpe, t)
        }
        case _         => Nil
      }
      /**
       * Get a type from the above mapping.
       * @param t Type whose substitution is required.
       * @return the type to substitute if any.
       */
      def fromMap(t: UType) = {
        t match {
          case tv: TVarID => getTypeFromMapping(smap, tv)
          case nt: NType  => nt.subst(smap)
          case t          => t
        }
      }
      // Do the actual substitution
      val substituted = methodSignature(
          that.typeParams map fromMap,
          that.isPure,
          fromMap(that.returnType),
          that.paramTypes map fromMap,
          that.sym
      )        
      this =~= (substituted,(a,b) => a <:< b) && (this.isPure == substituted.isPure || this.isPure)
    } 
  }

  /**
   * Factory method to create a new method signature from a method symbol.
   * @param sym Compiler symbol of the method.
   * @return the method signature if it was initialized, or a reason why this failed.
   */
  override def method(sym: Symbol) : ROption[MethodSignature] = {
    super.method(sym) match {
      case RSome(msig) if ! msig.isPure => 
        // Check if the owner's type is a value type (ie. Int, Boolean, etc.) but that its owner is
        // not a method (owner's type seems to be a value type for nested methods...), if the method 
        // is a getter generated by Scala, or if the method is otherwise specified/assumable
        // as pure (ie. through the pureMethods map).
        if ((sym.owner.tpe.isImmutable && ! sym.owner.isMethod)|| sym.isGetter || methodAssumableAsPure(sym)) {
          // Since the method is pure, arguments won't be modified anyway, so they 
          // should be any if not specified differently.
          val paramTypes = msig.paramTypes.map{
            case nt: NType if ! nt.tpe.isOwnershipAnnotated => nt.setOwnershipModifiers(any())
            case t:  UType                                  => t
          }
         
          val typeParams = msig.typeParams.map{
            case nt: NType if ! nt.tpe.isOwnershipAnnotated => nt.setOwnershipModifiers(any())
            case t:  UType                                => t
          }
          RSome(methodSignature(typeParams,true,msig.returnType,paramTypes,msig.sym))
        }
        else {
          RSome(msig)
        }
      case RSome(msig)                  => RSome(msig)
      case error                        => error
    }
  }
}
