/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: UniverseTypeRepresentationBase.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.common

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common._

/**
 * Base trait for the internal representation of Universe types. 
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait UniverseTypeRepresentationBase[G <: Global] {
  /**
   * Global instance used for access to the compiler's type representation.
   */
  val global : G
  import global._
  import UTSDefaults._

  /**
   * Logger for debugging output and warning/error handling.
   */
  val logger : UTSLogger
  
  /**
   * Extend the <code>Type</code> by several helper functions 
   * using an implicit conversion.
   */
  val extendedType = new ExtendedType {
    val global : UniverseTypeRepresentationBase.this.global.type = UniverseTypeRepresentationBase.this.global
  }
  import extendedType._
  
  /**
   * Base class for the internal representation of a type.
   */
  abstract class UType(val tpe: Type) {
    /**
     * Is this <code>UType</code> a type parameter? 
     */
    private var typeParameter = false
    
    /**
     * Return if this <code>UType</code> is a type parameter.
     * @return if this <code>UType</code> is a type parameter.
     */
    def isTypeParameter = typeParameter
      
    /**
     * Set this type to be a type parameter.
     */
    def setAsTypeParameter {
      typeParameter = true
    }
     
    /**
     * Compare two types concerning equality.
     * @param that The type to compare to.
     * @return if the types are equal.
     */
    def =:=(that: UType) : Boolean = false
      
    /**
     * Return if this type is <code>scala.Unit</code>.
     * @return if type is <code>scala.Unit</code>.
     */
    def isUnit : Boolean = {
      this.tpe.isUnit
    }
      
    /**
     * Adapt this type to an ownership modifier.
     * @param The ownership modifier this type should be adapted to.
     * @return the adapted type.
     */
    def |>:(om: OwnershipModifier) : UType
       
    /**
     * Limited covariance rule.
     * @param that The other type.
     * @return if the types are the same.
     */
    def ta(that: UType) : Boolean
      
    /**
     * Check if <code>that</code> is a supertype of <code>this</code>.
     * @param that Probable supertype.
     * @return if <code>that</code> is a supertype of <code>this</code>.
     */
    def <:<(that: UType) : Boolean
    
    /**
     * Check if the type contains the ownership modifier <code>u</code>.
     * @param u The ownership modifier to look for.
     * @return if the modifiers matched somewhere.
     */
    def contains(u: OwnershipModifier) : Boolean
     
    /**
     * Check the type for well-formedness.
     * @return a <code>TypeOption</code> containing the type if is was well-formed.
     */
    def check : UType
    
    /**
     * Owner-as-modifier: Check if modification of the referenced object 
     * is basically allowed.
     * @return if the modification is allowed.
     */
    def modificationOK : Boolean
    
    /**
     * Add a ownership modifier annotation to a <code>UType</code>.
     * @param t0 The <code>UType</code> which should get the annotation.
     * @return the modified <code>UType</code>
     */
    def addOwnershipModifiersOn(t0: UType) : UType
    
    /**
     * Upper type bound for this type. Upper bound of the underlying type in the
     * case of type variables, and <code>this</code> for non-variable types.
     * @return the upper bound of the type.
     */
    def ubgamma : UType
    
    /**
     * Check if the underlying type or its upperbound in the case of a type variable 
     * was ownership annotated.
     * @return if the underlying type was containd ownership modifier annotations.
     */
    def isUnderlyingOwnershipAnnotated : Boolean = {
      tpe.isOwnershipAnnotated || ubgamma.tpe.isOwnershipAnnotated
    }
  }
}
