/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: OwnershipModifierExtender.scala 817 2008-01-25 19:44:05Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.common

import ch.ethz.inf.sct.uts.annotation._

/**
 * Trait encapsulating an <code>ExtendedOwnershipModifier</code> which may 
 * be overwritten by subclasses in order to implement actual behavior of the 
 * modifiers.
 * 
 * @author  Manfred Stock
 * @version $Revision: 817 $
 */
trait OwnershipModifierExtender {
  /**
   * An extended variant of an ownership modifier which implements 
   * viewpoint adaptation and other functions.
   */
  abstract class ExtendedOwnershipModifier(val om: OwnershipModifier) {
    /**
     * Name of this ownership modifier.
     * @return the name of the encapsulated ownership modifier. 
     */
    def name = om.name
    
    /**
     * The parent ownership modifier. Used to model the subuniversing rules.
     * @return the parent ownership modifier.
     */
    def parent: OwnershipModifier

    /**
	 * Is this modifier a main modifier?
     * @return if the encapsulated modifier is a main modifier.
     */
    def isMainModifier : Boolean
     
    /**
     * Get parent ownership modifier fulfilling some predicate.
     * @param p   Predicate the parent ownership modifier should fulfill.
     * @param alt Alternative if no parent modifier fulfills the predicate.
     * @return    the parent modifier fulfilling the predicate or the alternative 
     *            if none is found.
     */
    def getParent(p: OwnershipModifier => Boolean, alt: OwnershipModifier) : OwnershipModifier = {
      var cur = this.om
      while (cur != null && ! p(cur)) {
        cur = cur.parent
      }
      if (cur != null) {
        cur
      }
      else {
        alt
      }
    }
       
    /**
     * Implement subuniversing: u <:< u' <=> u <:<sub>u</sub> u'
     * @param that The modifier which should be tested for being higher up in the 
     *             hierarchy than <code>this</code>.
     * @return if <code>this</code> is below <code>that</code>.
     */
    def <:< (that: OwnershipModifier) : Boolean = {
      getParent(p => p == that, null) != null
    }
       
    /**
     * Get the parent ownership modifier which is a main modifier.
     * @return the parent annotation of <code>this</code> which is a main modifier. 
     */
    def parentMainmodifier : OwnershipModifier = {
      getParent(p => p.isMainModifier, any())
    }
      
    /**
     * Get the parent ownership modifier which may appear as an annotation 
     * in the source code of the program.
     * @return the parent annotation of <code>this</code> which could be used in a program.
     */
    def parentOwnershipAnnotation : OwnershipModifier = {
      getParent(p => p.isInstanceOf[OwnershipModifierAnnotation], any())
    } 

    /**
     * Viewpoint adaptation of an ownership modifier w.r.t. an ownership modifier.
     * @param uprime Modifier which is to be adapted.
     * @return the adapted ownership modifier.
     */
    def |> (uprime: OwnershipModifier) : OwnershipModifier
  }
  
  /**
   * Implicitly convert an ownership modifier to an <code>ExtendedOwnershipModifier</code>.
   * @param om The normal ownership modifier.
   * @return the enriched ownership modifier
   */
  implicit def om2eom(om: OwnershipModifier) : ExtendedOwnershipModifier
}
