/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: ErrorTypes.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.staticcheck.common;

import scala.tools.nsc._
import ch.ethz.inf.sct.uts.annotation._
import ch.ethz.inf.sct.uts.plugin.common._

/**
 * Representation of error types. Used for error handling and reporting.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
trait ErrorTypes[G <: Global] {
  self: TypeAbstraction[G] => 
   
  import global._
  
  /**
   * Base class for type errors and error handling. Basically returns <code>true</code> for
   * all boolean methods since we already are in an invalid state and don't want to
   * trigger more error messages, as they probably don't make too much sense, anyway. 
   */
  sealed abstract class ErroneousType extends UType(NoType) {
    /**
     * Check the type for well-formedness. Since this type is invalid in any case, 
     * always returns <code>this</code>.
     * @return <code>this</code> since type is always invalid.
     */
    def check : UType = {
      this
    }
        
    /**
     * Dummy subtype check.
     */
    def <:<(that: UType) = true

    /**
     * Return always false, as usually the absence of ownership modifiers is checked.
     * @param u The ownership modifier to look for.
     * @return if the modifiers matched somewhere, which is never the case here.
     */
    def contains(u: OwnershipModifier) : Boolean = false
       
    /**
     * Dummy equality function.
     */
    override def =:=(that: UType) = true
       
    /**
     * Dummy ta function.
     */
    def ta(that: UType) = true

    /**
     * Owner-as-modifier: Check if modification of the referenced object 
     * is basically allowed. Always true for <code>InvalidType</code>s 
     * since they are invalid anyway.
     * @return if the modification is allowed.
     */
    def modificationOK : Boolean = true
      
    /**
     * Add an ownership modifier annotation to a <code>UType</code>. Nothing done here, as 
     * the <code>InvalidType</code> does not contain ownership modifiers.
     * @param t0 The <code>UType</code> which should get the annotation.
     * @return the same type which was given as argument.
     */
    def addOwnershipModifiersOn(t0: UType) : UType = {
      t0
    }
      
    /**
     * Adapt this type to an ownership modifier. Return <code>this</code> since 
     * we already are in an error state, so adaptation is futile.
     * @param The ownership modifier this type should be adapted to.
     * @return the adapted type.
     */
    def |>:(om: OwnershipModifier) : UType = this
    
    /**
     * Log the error.
     */
    def log
    
    /**
     * Dummy upper type bound for this type. 
     * @return <code>this</code>.
     */
    def ubgamma : UType = this
  }
   
  /**
   * Class to represent an invalid type. 
   * @param reason A reason why this is an invalid type.
   */
  final case class InvalidType(val reason: String) extends ErroneousType {
    /**
     * Position where the error occurred.
     */
    private var pos : util.Position = null
    
    /**
     * Set the position where the error occurred.
     * @param pos Position of error.
     * @return a new <code>InvalidType</code> but with a position.
     */
    def setPosition(pos: util.Position) : InvalidType = {
      new InvalidType(reason,pos)
    }
    
    /**
     * Get position of the error.
     * @return the position of the error.
     */
    def getPosition : util.Position = pos
     
    /**
     * Create a new <code>InvalidType</code> but with a position.
     * @param reason   Reason for the error.
     * @param position Position of the error.
     */
    def this(reason: String, position: util.Position) {
      this(reason)
      pos = position
    }

    /**
     * Get reason why type is invalid.
     * @return the reason why this type is invalid.
     */
    def getReason = if (reason != null) { reason } else { "" }
         
    /**
     * Convert invalid type to a string.
     * @return the string representation.
     */
    override def toString = {
      getReason
    }
    
    /**
     * Log the error.
     */
    def log {
      if (pos != null) {
        logger.error(reason,pos)
      }
      else {
        logger.error(getReason)
      }
    }
  }
  
  /**
   * Class to encapsulate several invalid types. 
   * @param errors The errors which should be encapsulated.
   */
  final case class InvalidTypes(val errors: List[InvalidType]) extends ErroneousType {
    /**
     * Convert invalid types to a string.
     * @return the string representation.
     */
    override def toString = {
      errors mkString ("","\n","")
    }
    
    /**
     * Log the errors.
     */
    def log {
      errors foreach { e => e.log }
    }
    
    /**
     * Append method to add another error.
     * @return a new <code>InvalidTypes</code> instance containing the new error.
     */
    def +(et: ErroneousType) = {
      et match {
        case it: InvalidType   => InvalidTypes(errors ::: List(it))
        case it: InvalidTypes  => InvalidTypes(errors ::: it.errors) 
      }
    }
    
    /**
     * Check if there are any errors encapsulated.
     * @return if errors are encapsulated.
     */
    def isEmpty = {
      errors.isEmpty
    }
    
    /**
     * If there are no errors encapsulated, return the given type, else
     * return <code>this</code>.
     * @param tp Type to return if no errors are encapsulated.
     * @return <code>this</code> if errors are encapsulated, <code>tp</code> else.
     */
    def orElse(tp: UType) : UType = {
      if (isEmpty) {
        tp
      }
      else {
        this
      }
    }
  }
}
