/*
 *              __  ____________        ____         __    
 *             / / / /_  __/ __/ ____  / __/______ _/ /__ _
 *            / /_/ / / / _\ \  /___/ _\ \/ __/ _ `/ / _ `/
 *            \____/ /_/ /___/       /___/\__/\_,_/_/\_,_/ 
 * 
 * This file is part of an implementation of the Universe Type System for
 * Scala.
 * 
 * Copyright (C) 2007-2008  Swiss Federal Institute of Technology, Zurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * 
 * $Id: MJRuntimeCheckTransform.scala 883 2008-02-01 18:59:56Z ms $
 */
package ch.ethz.inf.sct.uts.plugin.runtimecheck

import scala.tools.nsc.transform._
import scala.tools.nsc.util._
import scala.tools.nsc.symtab._
import ch.ethz.inf.sct.uts.plugin.common._
import ch.ethz.inf.sct.uts.annotation._
import org.multijava.mjc.MjcTokenTypes

/**
 * Transform the AST such that it contains runtime UTS checks afterwards.
 * 
 * @author  Manfred Stock
 * @version $Revision: 883 $
 */
abstract class MJRuntimeCheckTransform extends RuntimeCheckTransformBase {
  import global._
  import definitions._             // standard classes and methods
  import typer.{typed, atOwner}    // methods to type trees
  import posAssigner.atPos         // for filling in tree positions
  import UTSDefaults._
  import extendedType._
  
  /**
   * Factory method to create new <code>Transformer</code>s.
   * @param unit The unit which will be processed by the <code>Transformer</code>.
   * @return the new <code>Transformer</code>.
   */
  def newTransformer(unit: CompilationUnit) : Transformer =
    new RuntimeCheckTransformer(unit)
  
  /**
   * Transformer which adds UTS runtime checks to the AST. As it seems, this is done
   * using a depth-first traversal of the AST.
   * @param unit The unit where the checks should be added.
   */
  class RuntimeCheckTransformer(unit: CompilationUnit) extends RTCTransformer(unit) {
    val runtimeObject = UTSDefaults.mjRuntimeObject
    
    /**
     * Process the call of a constructor.
     * @param tree      Tree with the call. 
     * @param om        Main modifier of the newly created instance's type.
     * @param tmp       Temporary variable which contains the new instance.
     * @param tmpaccess Function which returns a tree for access to the temporary variable.
     * @return the body of the block which is to be added in the place of the constructor call.
     */
    def processConstructorCall(tree: Tree, owner: OwnershipModifier, tmp: Tree, tmpaccess: () => Ident) : List[Tree] = {
      tree match {
        case Apply(fun@Select(qualifier@New(tpt),selector),args) =>
          List(
              rthMethod(
                  "setConstructorData",
                  List(
                      This(currentOwner.enclClass),
                      getClassFromType(tpt.tpe),
                      owner match {
                        case peer() => Literal(MjcTokenTypes.LITERAL_peer)
                        case rep()  => Literal(MjcTokenTypes.LITERAL_rep)
                      }
                  )
              ),
              tmp,
              rthMethod(
                  owner match {
                    case rep()  => "setOwnerRep"
                    case peer() => "setOwnerPeer"
                  },
                  List(
                      tmpaccess(),
                      This(currentOwner.enclClass)
                  )
              )
          )
      }
    }
    
    /**
     * Process a call of the <code>isInstanceOf</code>/<code>asInstanceOf</code> method.
     * @param tree    The tree containing the call.
     * @param mainmod Main modifier of the type the object is checked for/cast to.
     * @return the tree which executes the additional check.
     */
    def processIsAsInstanceOf(tree: Tree, mainmod: OwnershipModifier) : Tree = {
      tree match {
        case ta @ TypeApply(sel @ Select(qualifier,selector),tpelist) =>
          // Call to the method of the runtime support class
          val runtimeCall = rthMethod(
              mainmod match {
                case peer() => "isPeer"
                case rep()  => "isOwner"
              },
              List(
                  This(currentOwner.enclClass),
                  qualifier match {
                    case Select(qual,sel) => Select(qual,sel) setSymbol qualifier.symbol
                    case Ident(id)        => Ident(id) setSymbol qualifier.symbol
                    case _                => qualifier
                  }
              )
          )

          // Create the runtime checks
          if (sel.symbol == Any_isInstanceOf) {
            logger.debug("Adding runtime check for isInstanceOf...")
            typed(atPos(tree.pos)(
                Apply(
                    Select(
                        ta,
                        nme.AMPAMP
                    ),
                    List(
                        runtimeCall
                    )
                )
            ))
          }
          else if (sel.symbol == Any_asInstanceOf) {
            logger.debug("Adding runtime check for asInstanceOf...")
            typed(atPos(tree.pos)(
                Block(
                    List(
                        If(
                            Select(
                                runtimeCall,
                                nme.UNARY_!
                            ),
                            rtpMethod("illegalCast",Nil),
                            EmptyTree
                        )
                    ), 
                    ta
                )
            ))
          }
          else {
            tree
          }
      }
    }
  }
}
