/* Kernel module to match aodv-extensions as defined in RFC3561
 * p. 30 which is also switchable from userspace using setsockopt */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/skbuff.h>
#include <linux/if_ether.h>
#include <linux/ip.h>
#include <linux/udp.h>

#include <asm/semaphore.h>

#include <linux/netfilter_ipv4/ip_tables.h>

#include "ipt_aodvext.h"

/* Constants for aodv extensions */
#define RREQSIZE 24
#define RREPSIZE 20
#define UDPHEADSIZE 8
#define MINIPHEADSIZE 20

#define RREQ 1
#define RREP 2

#define HADEXT 1
#define NOEXT 0

#if 0
#define DEBUGP printk
#else
#define DEBUGP(format, args...)
#endif

/* some sort of reference counter for ruleswitcher */
static int accept = 0;
/* mutex for write access to accept */
static DECLARE_MUTEX(accept_lock);
/* counter for hellos */
static int hellocount = 0;
/* mutex for write access to hellocount */
static DECLARE_MUTEX(hellocount_lock);

/* Do some checks to find out if the packet *could* be an AODV-packet
   and set some variables */
static int probablyAODV(const struct sk_buff *skb, const struct iphdr *iph, int *coffset, __u16 *udplen){
  // Check that length is larger or equal to the minimal length of a
  // rreq/rrep (20 byte IP, 8 byte UDP, 20 Byte rrep (which is 4 byte
  // shorter than rreq))
  if(skb->len >= 20+UDPHEADSIZE+RREPSIZE){
    if((void*)iph == (void*)skb->data){
      *coffset = iph->ihl*4;
      // Make sure that the current offset is valid and allows for an
      // aodv-rrep-packet to be in the udp-packet
      if(*coffset >= MINIPHEADSIZE && *coffset <= skb->len-UDPHEADSIZE-RREPSIZE){
	*udplen = skb->data[*coffset+4] << 8 | skb->data[*coffset+5];
	DEBUGP(KERN_DEBUG "UDP-packet: %i byte (0x%hx byte).\n",*udplen,*udplen);
	// Check if udp-length is valid, and large enough to contain an
	// aodv-rreq/rrep-packet (at least 20 byte)
	if(*udplen+*coffset == skb->len && *udplen >= UDPHEADSIZE+RREPSIZE){
	  *coffset += UDPHEADSIZE;
	  return 1;
	}else{
	  DEBUGP(KERN_DEBUG "Length of UDP-packet too small for containing an AODV-packet.\n");
	}
      }else{
	DEBUGP(KERN_DEBUG "Content of IP-packet too small for containing AODV-packet.\n");
      }
    }else{
      DEBUGP(KERN_DEBUG "Data does not start with ip-header.\n");
    }
  }else{
    DEBUGP(KERN_DEBUG "Packet too small (%d) to contain AODV-packet.\n",skb->len);
  }
  return 0;
} 

/* Print info on RREQ */
static void rreqinfo(const struct sk_buff *skb, const struct iphdr *iph, int offset, char where, __u8 hasext){
  __u8 hopcount = skb->data[offset+3];
  __u32 *data = (__u32*) skb->data+(offset/4);
  printk(KERN_DEBUG "%c:AODV RREQ: SRC: %u.%u.%u.%u; DST: %u.%u.%u.%u; Hopcount: %u; "
	 "DST-IP: %u.%u.%u.%u; DST-SEQ: %u; "
	 "ORIG-IP: %u.%u.%u.%u; ORIG-SEQ: %u; RREQ-ID: %u; Has extension: %u\n", 
	 where, NIPQUAD(iph->saddr), NIPQUAD(iph->daddr), hopcount,
	 NIPQUAD(data[2]), ntohl(data[3]), 
	 NIPQUAD(data[4]), ntohl(data[5]), ntohl(data[1]), hasext);
}

/* Print info on RREP */
static void rrepinfo(const struct sk_buff *skb, const struct iphdr *iph, int offset, char where, __u8 hasext){
  __u8 hopcount = skb->data[offset+3];
  __u32 *data = (__u32*) skb->data+(offset/4);
  printk(KERN_DEBUG "%c:AODV RREP: SRC: %u.%u.%u.%u; DST: %u.%u.%u.%u; Hopcount: %u; "
	 "DST-IP: %u.%u.%u.%u; DST-SEQ: %u; "
	 "ORIG-IP: %u.%u.%u.%u; Lifetime: %u; Has extension: %u\n", 
	 where, NIPQUAD(iph->saddr), NIPQUAD(iph->daddr), hopcount,
	 NIPQUAD(data[1]),ntohl(data[2]),
	 NIPQUAD(data[3]),ntohl(data[4]), hasext);
}

/* Print some information about the (AODV-) packet. */
static void aodvinfo(const struct sk_buff *skb, char where){
  const struct iphdr *iph = skb->nh.iph;
  int coffset = 0;
  __u16 udplen = 0;
  __u8 type = 0;

  if(probablyAODV(skb,iph,&coffset,&udplen)){
    type = skb->data[coffset];
    // Check if packet is larger than necessary for given type -
    // if yes, assume it contains an extension.
    switch(type){
    case RREQ:
      DEBUGP(KERN_DEBUG "AODV-RREQ-Packet.\n");
      if(udplen >= UDPHEADSIZE+RREQSIZE){
	if(udplen > UDPHEADSIZE+RREQSIZE){
	  rreqinfo(skb,iph,coffset,where,1);
	}else{
	  rreqinfo(skb,iph,coffset,where,0);
	}
      }
      break;
    case RREP:
      DEBUGP(KERN_DEBUG "AODV-RREP-Packet.\n");
      if(udplen >= UDPHEADSIZE+RREPSIZE){
	if(udplen > UDPHEADSIZE+RREPSIZE){
	  rrepinfo(skb,iph,coffset,where,1);
	}else{
	  rrepinfo(skb,iph,coffset,where,0);
	}
      }
      break;
    }
  }
}

/* The match function. Returns nonzero if the packet matches. */
static int
match(const struct sk_buff *skb,
      const struct net_device *in,
      const struct net_device *out,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,17)
	  const struct xt_match *match,
#endif
      const void *matchinfo,
      int offset,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
      unsigned int protoff,
#endif
      int *hotdrop)
{
  const struct ipt_aodvext_info *info = matchinfo;
  __u16 udplen = 0;
  __u8 type = 0;
  int coffset = 0;
  const struct iphdr *iph = skb->nh.iph;

  if(info->alldebug){
    aodvinfo(skb,'A');
  }

  if(info->aodvext){
    DEBUGP(KERN_DEBUG "Packet Size: %i\n", skb->len);
  
    if(probablyAODV(skb,iph,&coffset,&udplen)){
      type = skb->data[coffset];
      // Check if packet is larger than necessary for given type -
      // if yes, assume it contains an extension.
      switch(type){
      case RREQ:
	DEBUGP(KERN_DEBUG "AODV-RREQ-Packet.\n");
	if(udplen-UDPHEADSIZE > RREQSIZE){
	  if(!info->alldebug && info->matchdebug && HADEXT ^ info->invert){
	    aodvinfo(skb,'E');
	  }
	  return HADEXT ^ info->invert;
	}
	break;
      case RREP:
	DEBUGP(KERN_DEBUG "AODV-RREP-Packet.\n");
	if(udplen-UDPHEADSIZE > RREPSIZE){
	  if(!info->alldebug && info->matchdebug && HADEXT ^ info->invert){
	    aodvinfo(skb,'E');
	  }
	  return HADEXT ^ info->invert;
	}
	break;
      }
      DEBUGP(KERN_DEBUG "UDP-Packet too short (%d) for containing aodv-extension or invalid type.\n",udplen); 
    }
    return NOEXT ^ info->invert;
  }else{
    if((accept > 0) ^ info->invert){
      /* Packets currently should be matched. */
      if(!info->alldebug && info->matchdebug){
	aodvinfo(skb,'T');
      }
      return (accept > 0) ^ info->invert;
    }else{
      /* Check if it could be an AODV-packet and match if it's a
	 HELLO-message (and they should be matched). */
      if(info->hello >= 0 && probablyAODV(skb,iph,&coffset,&udplen)){
	type = skb->data[coffset];
	/* Acquire lock */
	if (down_interruptible(&hellocount_lock))
	  return -EINTR;
	/* Check if it's an AODV-HELLO-message => type RREP,
	   Hopcount=0, TTL=1. Also make sure destination address and
	   originator address are the same. */
	if(type == RREP && 
	   skb->data[coffset+3] == 0 && 
	   iph->ttl == 1 && 
	   memcmp(&(skb->data[coffset+4]),&(skb->data[coffset+12]),4) == 0){
	  if(hellocount >= info->hello){
	    hellocount = 1;
	    if(!info->alldebug && info->matchdebug){
	      aodvinfo(skb,'H');
	    }
	    up(&hellocount_lock);
	    return ~info->invert;
	  }else{
	    hellocount++;
	  }
	}
	up(&hellocount_lock);
      }
      return info->invert;
    }
  }
}

static int
ipt_aodvext_checkentry(const char *tablename,
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,16)
		       const struct ipt_ip *ip,
#else
		       const void *ipp,
#endif
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,17)
			   const struct xt_match *match,
#endif
		       void *matchinfo,
		       unsigned int matchsize,
		       unsigned int hook_mask)

{
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,16)
  const struct ipt_ip *ip = (const struct ipt_ip *) ipp;
#endif
  if (ip->proto != IPPROTO_UDP || (ip->invflags & IPT_INV_PROTO)) {
    printk(KERN_ERR "aodvext: Only works on UDP packets (use -p udp).\n");
    return 0;
  }
  if(accept < 0){
    printk(KERN_ERR "accept should be >= 0!\n");
    return 0;
  }
  return 1;
}

static int
do_set_accept(struct sock *sk, int cmd, void __user *user, unsigned int len)
{
  int value;

  if (!capable(CAP_NET_ADMIN))
    return -EPERM;

  if (copy_from_user(&value, user, sizeof(value)) != 0)
    return -EFAULT;
  
  /* Acquire lock */
  if (down_interruptible(&accept_lock))
    return -EINTR;
  
  DEBUGP(KERN_DEBUG "Value before: %i.\n",accept);
  
  if(value == RS_ON){
    accept += 1;
  }else if(value == RS_OFF){
    if(accept > 0)
      accept -= 1;
  }else{
    printk(KERN_WARNING "%s:%i: Invalid value %i!\n",__FILE__,__LINE__,value);
    up(&accept_lock);
    return -ENOPROTOOPT;
  }
  
  DEBUGP(KERN_DEBUG "Value after: %i\n",accept);
  up(&accept_lock);
  return 0;
}

static int
do_get_accept(struct sock *sk, int cmd, void __user *user, int *len){
  if(*len < sizeof(accept)){
    printk(KERN_WARNING "%s:%i: Not enough space provided - %i words necessary\n",__FILE__,__LINE__,sizeof(accept));
    return -EFAULT;
  }
  
  if (copy_to_user(user, &accept, sizeof(accept)) != 0)
    return -EFAULT;
  
  return 0;
}

static struct nf_sockopt_ops aodvext_sockopt_ops = {
  .list = {NULL,NULL},
  .pf = PF_INET,
  .set_optmin = RS_SOCKOPTNUM,
  .set_optmax = RS_SOCKOPTNUM+1,
  .set = do_set_accept,
  .get_optmin = RS_SOCKOPTNUM,
  .get_optmax = RS_SOCKOPTNUM+1,
  .get = do_get_accept,
};

static struct ipt_match aodvext_match = {
  .name           = "aodvext",
  .match          = &match,
  .checkentry     = &ipt_aodvext_checkentry,
  .me             = THIS_MODULE,
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,17)
  .matchsize      = sizeof(struct ipt_aodvext_info)
#endif
};

static int __init init(void)
{
  int ret;
  if((ret = nf_register_sockopt(&aodvext_sockopt_ops)) < 0){
    printk(KERN_ERR "Unable to register sockopts.\n");
    return ret;
  }
  DEBUGP(KERN_DEBUG "Trying to load match for aodv-extensions...\n");
  return ipt_register_match(&aodvext_match);
}

static void __exit fini(void)
{
  nf_unregister_sockopt(&aodvext_sockopt_ops);
  ipt_unregister_match(&aodvext_match);
  DEBUGP(KERN_DEBUG "Match for aodv-extensions unloaded\n");
}

module_init(init);
module_exit(fini);
